import { jsx, jsxs, Fragment } from 'react/jsx-runtime';
import 'react';
import { useQueryParams, useNotification, Page, RESPONSIVE_DEFAULT_SPACING, tours, Form, useRBAC } from '@strapi/admin/strapi-admin';
import { Tabs, Main, Box, Grid } from '@strapi/design-system';
import { useIntl } from 'react-intl';
import { useLocation, useParams } from 'react-router-dom';
import { styled } from 'styled-components';
import { SINGLE_TYPES } from '../../constants/collections.mjs';
import { PERMISSIONS } from '../../constants/plugin.mjs';
import { useDocumentRBAC, DocumentRBAC } from '../../features/DocumentRBAC.mjs';
import { useDoc } from '../../hooks/useDocument.mjs';
import { useDocumentLayout } from '../../hooks/useDocumentLayout.mjs';
import { useLazyComponents } from '../../hooks/useLazyComponents.mjs';
import { useOnce } from '../../hooks/useOnce.mjs';
import { getTranslation } from '../../utils/translations.mjs';
import { createYupSchema } from '../../utils/validation.mjs';
import { Blocker } from './components/Blocker.mjs';
import { FormLayout } from './components/FormLayout.mjs';
import { Header } from './components/Header.mjs';
import { Panels } from './components/Panels.mjs';
import { handleInvisibleAttributes } from './utils/data.mjs';

/* -------------------------------------------------------------------------------------------------
 * EditViewPage
 * -----------------------------------------------------------------------------------------------*/ const EditViewPage = ()=>{
    const location = useLocation();
    const [{ query: { status } }, setQuery] = useQueryParams({
        status: 'draft'
    });
    const { formatMessage } = useIntl();
    const { toggleNotification } = useNotification();
    const doc = useDoc();
    const { document, meta, isLoading: isLoadingDocument, schema, components, collectionType, id, model, hasError, getTitle, getInitialFormValues } = doc;
    const hasDraftAndPublished = schema?.options?.draftAndPublish ?? false;
    useOnce(()=>{
        /**
     * We only ever want to fire the notification once otherwise
     * whenever the app re-renders it'll pop up regardless of
     * what we do because the state comes from react-router-dom
     */ if (location?.state && 'error' in location.state) {
            toggleNotification({
                type: 'danger',
                message: location.state.error,
                timeout: 5000
            });
        }
    });
    const isLoadingActionsRBAC = useDocumentRBAC('EditViewPage', (state)=>state.isLoading);
    const isSingleType = collectionType === SINGLE_TYPES;
    /**
   * single-types don't current have an id, but because they're a singleton
   * we can simply use the update operation to continuously update the same
   * document with varying params.
   */ const isCreatingDocument = !id && !isSingleType;
    const { isLoading: isLoadingLayout, edit: { layout, settings: { mainField } } } = useDocumentLayout(model);
    const pageTitle = getTitle(mainField);
    const { isLazyLoading } = useLazyComponents([]);
    const isLoading = isLoadingActionsRBAC || isLoadingDocument || isLoadingLayout || isLazyLoading;
    const initialValues = getInitialFormValues(isCreatingDocument);
    if (isLoading && !document?.documentId) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    if (!initialValues || hasError) {
        return /*#__PURE__*/ jsx(Page.Error, {});
    }
    const handleTabChange = (status)=>{
        if (status === 'published' || status === 'draft') {
            setQuery({
                status
            }, 'push', true);
        }
    };
    const validateSync = (values, options)=>{
        const yupSchema = createYupSchema(schema?.attributes, components, {
            status,
            ...options
        });
        return yupSchema.validateSync(values, {
            abortEarly: false
        });
    };
    return /*#__PURE__*/ jsxs(Main, {
        paddingLeft: RESPONSIVE_DEFAULT_SPACING,
        paddingRight: RESPONSIVE_DEFAULT_SPACING,
        children: [
            /*#__PURE__*/ jsx(Page.Title, {
                children: pageTitle
            }),
            isSingleType && /*#__PURE__*/ jsx(tours.contentManager.Introduction, {
                children: /*#__PURE__*/ jsx(Box, {})
            }),
            /*#__PURE__*/ jsx(Form, {
                disabled: hasDraftAndPublished && status === 'published',
                initialValues: initialValues,
                method: isCreatingDocument ? 'POST' : 'PUT',
                validate: (values, options)=>{
                    // removes hidden fields from the validation
                    // this is necessary because the yup schema doesn't know about the visibility conditions
                    // and we don't want to validate fields that are not visible
                    const { data: cleanedValues, removedAttributes } = handleInvisibleAttributes(values, {
                        schema,
                        initialValues,
                        components
                    });
                    const yupSchema = createYupSchema(schema?.attributes, components, {
                        status,
                        removedAttributes,
                        ...options
                    });
                    return yupSchema.validate(cleanedValues, {
                        abortEarly: false
                    });
                },
                initialErrors: location?.state?.forceValidation ? validateSync(initialValues, {}) : {},
                children: /*#__PURE__*/ jsxs(Fragment, {
                    children: [
                        /*#__PURE__*/ jsx(Header, {
                            isCreating: isCreatingDocument,
                            status: hasDraftAndPublished ? getDocumentStatus(document, meta) : undefined,
                            title: pageTitle
                        }),
                        /*#__PURE__*/ jsxs(Tabs.Root, {
                            variant: "simple",
                            value: status,
                            onValueChange: handleTabChange,
                            children: [
                                /*#__PURE__*/ jsx(Tabs.List, {
                                    "aria-label": formatMessage({
                                        id: getTranslation('containers.edit.tabs.label'),
                                        defaultMessage: 'Document status'
                                    }),
                                    children: hasDraftAndPublished ? /*#__PURE__*/ jsxs(Fragment, {
                                        children: [
                                            /*#__PURE__*/ jsx(StatusTab, {
                                                value: "draft",
                                                children: formatMessage({
                                                    id: getTranslation('containers.edit.tabs.draft'),
                                                    defaultMessage: 'draft'
                                                })
                                            }),
                                            /*#__PURE__*/ jsx(StatusTab, {
                                                disabled: !meta || meta.availableStatus.length === 0,
                                                value: "published",
                                                children: formatMessage({
                                                    id: getTranslation('containers.edit.tabs.published'),
                                                    defaultMessage: 'published'
                                                })
                                            })
                                        ]
                                    }) : null
                                }),
                                /*#__PURE__*/ jsxs(Grid.Root, {
                                    paddingTop: {
                                        initial: 2,
                                        medium: 4,
                                        large: 8
                                    },
                                    gap: 4,
                                    children: [
                                        /*#__PURE__*/ jsxs(Grid.Item, {
                                            col: 9,
                                            xs: 12,
                                            direction: "column",
                                            alignItems: "stretch",
                                            children: [
                                                /*#__PURE__*/ jsxs(Tabs.Content, {
                                                    value: "draft",
                                                    children: [
                                                        /*#__PURE__*/ jsx(tours.contentManager.Fields, {
                                                            children: /*#__PURE__*/ jsx(Box, {})
                                                        }),
                                                        /*#__PURE__*/ jsx(FormLayout, {
                                                            layout: layout,
                                                            document: doc
                                                        })
                                                    ]
                                                }),
                                                /*#__PURE__*/ jsx(Tabs.Content, {
                                                    value: "published",
                                                    children: /*#__PURE__*/ jsx(FormLayout, {
                                                        layout: layout,
                                                        document: doc
                                                    })
                                                })
                                            ]
                                        }),
                                        /*#__PURE__*/ jsx(Grid.Item, {
                                            col: 3,
                                            xs: 12,
                                            direction: "column",
                                            alignItems: "stretch",
                                            children: /*#__PURE__*/ jsx(Panels, {})
                                        })
                                    ]
                                })
                            ]
                        }),
                        /*#__PURE__*/ jsx(Blocker, {})
                    ]
                })
            })
        ]
    });
};
const StatusTab = styled(Tabs.Trigger)`
  text-transform: uppercase;
`;
/**
 * @internal
 * @description Returns the status of the document where its latest state takes priority,
 * this typically will be "published" unless a user has edited their draft in which we should
 * display "modified".
 */ const getDocumentStatus = (document, meta)=>{
    const docStatus = document?.status;
    const statuses = meta?.availableStatus ?? [];
    /**
   * Creating an entry
   */ if (!docStatus) {
        return 'draft';
    }
    /**
   * We're viewing a draft, but the document could have a published version
   */ if (docStatus === 'draft' && statuses.find((doc)=>doc.publishedAt !== null)) {
        return 'published';
    }
    return docStatus;
};
/* -------------------------------------------------------------------------------------------------
 * ProtectedEditViewPage
 * -----------------------------------------------------------------------------------------------*/ const ProtectedEditViewPage = ()=>{
    const { slug = '' } = useParams();
    const { permissions = [], isLoading, error } = useRBAC(PERMISSIONS.map((action)=>({
            action,
            subject: slug
        })));
    if (isLoading) {
        return /*#__PURE__*/ jsx(Page.Loading, {});
    }
    if (error || !slug) {
        return /*#__PURE__*/ jsx(Page.Error, {});
    }
    return /*#__PURE__*/ jsx(Page.Protect, {
        permissions: permissions,
        children: ({ permissions })=>/*#__PURE__*/ jsx(DocumentRBAC, {
                permissions: permissions,
                children: /*#__PURE__*/ jsx(EditViewPage, {})
            })
    });
};

export { EditViewPage, ProtectedEditViewPage, getDocumentStatus };
//# sourceMappingURL=EditViewPage.mjs.map
