import { jsxs, jsx } from 'react/jsx-runtime';
import * as React from 'react';
import { useTracking, useNotification, useQueryParams, useRBAC, Layouts, GradientBadge, ConfirmDialog } from '@strapi/admin/strapi-admin';
import { Dialog, Typography, Link, Button, Flex } from '@strapi/design-system';
import { ArrowLeft, WarningCircle } from '@strapi/icons';
import { stringify } from 'qs';
import { useIntl } from 'react-intl';
import { useNavigate, useParams, NavLink } from 'react-router-dom';
import { PERMISSIONS } from '../../constants/plugin.mjs';
import { useHistoryContext } from '../pages/History.mjs';
import { useRestoreVersionMutation } from '../services/historyVersion.mjs';

const VersionHeader = ({ headerId })=>{
    const [isConfirmDialogOpen, setIsConfirmDialogOpen] = React.useState(false);
    const navigate = useNavigate();
    const { formatMessage, formatDate } = useIntl();
    const { trackUsage } = useTracking();
    const { toggleNotification } = useNotification();
    const [{ query }] = useQueryParams();
    const { collectionType, slug } = useParams();
    const [restoreVersion, { isLoading }] = useRestoreVersionMutation();
    const { allowedActions } = useRBAC(PERMISSIONS.map((action)=>({
            action,
            subject: slug
        })));
    const version = useHistoryContext('VersionHeader', (state)=>state.selectedVersion);
    const mainField = useHistoryContext('VersionHeader', (state)=>state.mainField);
    const schema = useHistoryContext('VersionHeader', (state)=>state.schema);
    const isCurrentVersion = useHistoryContext('VersionHeader', (state)=>state.page === 1 && state.versions.data[0].id === state.selectedVersion.id);
    const mainFieldValue = version.data[mainField];
    const getNextNavigation = ()=>{
        const pluginsQueryParams = stringify({
            plugins: query.plugins
        }, {
            encode: false
        });
        return {
            pathname: '..',
            search: pluginsQueryParams
        };
    };
    const handleRestore = async ()=>{
        try {
            const response = await restoreVersion({
                documentId: version.relatedDocumentId,
                collectionType,
                params: {
                    versionId: version.id,
                    contentType: version.contentType
                },
                body: {
                    contentType: version.contentType
                }
            });
            if ('data' in response) {
                navigate(getNextNavigation(), {
                    relative: 'path'
                });
                toggleNotification({
                    type: 'success',
                    title: formatMessage({
                        id: 'content-manager.restore.success.title',
                        defaultMessage: 'Version restored.'
                    }),
                    message: formatMessage({
                        id: 'content-manager.restore.success.message',
                        defaultMessage: 'A past version of the content was restored.'
                    })
                });
                trackUsage('didRestoreHistoryVersion');
            }
            if ('error' in response) {
                toggleNotification({
                    type: 'danger',
                    message: formatMessage({
                        id: 'content-manager.history.restore.error.message',
                        defaultMessage: 'Could not restore version.'
                    })
                });
            }
        } catch (error) {
            toggleNotification({
                type: 'danger',
                message: formatMessage({
                    id: 'notification.error',
                    defaultMessage: 'An error occurred'
                })
            });
        }
    };
    return /*#__PURE__*/ jsxs(Dialog.Root, {
        open: isConfirmDialogOpen,
        onOpenChange: setIsConfirmDialogOpen,
        children: [
            /*#__PURE__*/ jsx(Layouts.BaseHeader, {
                id: headerId,
                title: formatDate(new Date(version.createdAt), {
                    year: 'numeric',
                    month: 'numeric',
                    day: 'numeric',
                    hour: 'numeric',
                    minute: 'numeric'
                }),
                secondaryAction: /*#__PURE__*/ jsx(GradientBadge, {
                    label: formatMessage({
                        id: 'components.premiumFeature.title',
                        defaultMessage: 'Premium feature'
                    })
                }),
                subtitle: /*#__PURE__*/ jsx(Typography, {
                    variant: "epsilon",
                    textColor: "neutral600",
                    children: formatMessage({
                        id: 'content-manager.history.version.subtitle',
                        defaultMessage: '{hasLocale, select, true {{subtitle}, in {locale}} other {{subtitle}}}'
                    }, {
                        hasLocale: Boolean(version.locale),
                        subtitle: `${mainFieldValue || ''} (${schema.info.singularName})`.trim(),
                        locale: version.locale?.name
                    })
                }),
                navigationAction: /*#__PURE__*/ jsx(Link, {
                    startIcon: /*#__PURE__*/ jsx(ArrowLeft, {}),
                    tag: NavLink,
                    to: getNextNavigation(),
                    relative: "path",
                    isExternal: false,
                    children: formatMessage({
                        id: 'global.back',
                        defaultMessage: 'Back'
                    })
                }),
                sticky: false,
                primaryAction: /*#__PURE__*/ jsx(Dialog.Trigger, {
                    children: /*#__PURE__*/ jsx(Button, {
                        disabled: !allowedActions.canUpdate || isCurrentVersion,
                        onClick: ()=>{
                            setIsConfirmDialogOpen(true);
                        },
                        children: formatMessage({
                            id: 'content-manager.history.restore.confirm.button',
                            defaultMessage: 'Restore'
                        })
                    })
                })
            }),
            /*#__PURE__*/ jsx(ConfirmDialog, {
                onConfirm: handleRestore,
                endAction: /*#__PURE__*/ jsx(Button, {
                    fullWidth: true,
                    variant: "secondary",
                    onClick: handleRestore,
                    loading: isLoading,
                    children: formatMessage({
                        id: 'content-manager.history.restore.confirm.button',
                        defaultMessage: 'Restore'
                    })
                }),
                children: /*#__PURE__*/ jsxs(Flex, {
                    direction: "column",
                    alignItems: "center",
                    justifyContent: "center",
                    gap: 2,
                    textAlign: "center",
                    children: [
                        /*#__PURE__*/ jsx(Flex, {
                            justifyContent: "center",
                            children: /*#__PURE__*/ jsx(WarningCircle, {
                                width: "24px",
                                height: "24px",
                                fill: "danger600"
                            })
                        }),
                        /*#__PURE__*/ jsx(Typography, {
                            children: formatMessage({
                                id: 'content-manager.history.restore.confirm.title',
                                defaultMessage: 'Are you sure you want to restore this version?'
                            })
                        }),
                        /*#__PURE__*/ jsx(Typography, {
                            children: formatMessage({
                                id: 'content-manager.history.restore.confirm.message',
                                defaultMessage: "{isDraft, select, true {The restored content will override your draft.} other {The restored content won't be published, it will override the draft and be saved as pending changes. You'll be able to publish the changes at anytime.}}"
                            }, {
                                isDraft: version.status === 'draft'
                            })
                        })
                    ]
                })
            })
        ]
    });
};

export { VersionHeader };
//# sourceMappingURL=VersionHeader.mjs.map
