'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var strapiAdmin = require('@strapi/admin/strapi-admin');
var designSystem = require('@strapi/design-system');
var Icons = require('@strapi/icons');
var reactIntl = require('react-intl');
var reactRouterDom = require('react-router-dom');
var styledComponents = require('styled-components');
var DocumentStatus = require('../pages/EditView/components/DocumentStatus.js');
var homepage = require('../services/homepage.js');
var RelativeTime = require('./RelativeTime.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

const BASE_MAX_WIDTH = '14.4rem';
/**
 * Calculate dynamic max-width based on column span
 * Base width is 14.4rem for 6 columns, scale proportionally
 */ const calculateDynamicMaxWidth = (columnWidth = 4)=>{
    const baseColumnWidth = 4;
    const baseMaxWidth = 14.4; // rem
    const calculatedWidth = baseMaxWidth * columnWidth / baseColumnWidth;
    return `${Math.round(calculatedWidth * 10) / 10}rem`;
};
const CellTypography = styledComponents.styled(designSystem.Typography)`
  display: block;
  max-width: ${({ $maxWidth })=>$maxWidth || BASE_MAX_WIDTH};
  overflow: hidden;
  text-overflow: ellipsis;
  white-space: nowrap;
`;
const RecentDocumentsTable = ({ documents, type, dynamicMaxWidth = BASE_MAX_WIDTH })=>{
    const { formatMessage } = reactIntl.useIntl();
    const { trackUsage } = strapiAdmin.useTracking();
    const navigate = reactRouterDom.useNavigate();
    const getEditViewLink = (document)=>{
        const isSingleType = document.kind === 'singleType';
        const kindPath = isSingleType ? 'single-types' : 'collection-types';
        const queryParams = document.locale ? `?plugins[i18n][locale]=${document.locale}` : '';
        return `/content-manager/${kindPath}/${document.contentTypeUid}${isSingleType ? '' : '/' + document.documentId}${queryParams}`;
    };
    const handleRowClick = (document)=>()=>{
            trackUsage('willEditEntryFromHome', {
                entryType: type
            });
            const link = getEditViewLink(document);
            navigate(link);
        };
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Table, {
        colCount: 5,
        rowCount: documents?.length ?? 0,
        children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Tbody, {
            children: documents?.map((document)=>/*#__PURE__*/ jsxRuntime.jsxs(designSystem.Tr, {
                    onClick: handleRowClick(document),
                    cursor: "pointer",
                    children: [
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Td, {
                            children: /*#__PURE__*/ jsxRuntime.jsx(CellTypography, {
                                title: document.title,
                                variant: "omega",
                                textColor: "neutral800",
                                $maxWidth: dynamicMaxWidth,
                                children: document.title
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Td, {
                            children: /*#__PURE__*/ jsxRuntime.jsx(CellTypography, {
                                variant: "omega",
                                textColor: "neutral600",
                                $maxWidth: dynamicMaxWidth,
                                children: document.kind === 'singleType' ? formatMessage({
                                    id: 'content-manager.widget.last-edited.single-type',
                                    defaultMessage: 'Single-Type'
                                }) : formatMessage({
                                    id: document.contentTypeDisplayName,
                                    defaultMessage: document.contentTypeDisplayName
                                })
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Td, {
                            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                display: "inline-block",
                                children: document.status ? /*#__PURE__*/ jsxRuntime.jsx(DocumentStatus.DocumentStatus, {
                                    status: document.status
                                }) : /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                    textColor: "neutral600",
                                    "aria-hidden": true,
                                    children: "-"
                                })
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Td, {
                            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                textColor: "neutral600",
                                children: /*#__PURE__*/ jsxRuntime.jsx(RelativeTime.RelativeTime, {
                                    timestamp: new Date(document.updatedAt)
                                })
                            })
                        }),
                        /*#__PURE__*/ jsxRuntime.jsx(designSystem.Td, {
                            onClick: (e)=>e.stopPropagation(),
                            children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                display: "inline-block",
                                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.IconButton, {
                                    tag: reactRouterDom.Link,
                                    to: getEditViewLink(document),
                                    onClick: ()=>trackUsage('willEditEntryFromHome', {
                                            type
                                        }),
                                    label: formatMessage({
                                        id: 'content-manager.actions.edit.label',
                                        defaultMessage: 'Edit'
                                    }),
                                    variant: "ghost",
                                    children: /*#__PURE__*/ jsxRuntime.jsx(Icons.Pencil, {})
                                })
                            })
                        })
                    ]
                }, document.documentId))
        })
    });
};
/* -------------------------------------------------------------------------------------------------
 * LastEditedWidget
 * -----------------------------------------------------------------------------------------------*/ const LastEditedWidget = ({ columnWidth = 6 })=>{
    const { formatMessage } = reactIntl.useIntl();
    const { data, isLoading, error } = homepage.useGetRecentDocumentsQuery({
        action: 'update'
    });
    const dynamicMaxWidth = calculateDynamicMaxWidth(columnWidth);
    if (isLoading) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.Loading, {});
    }
    if (error || !data) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.Error, {});
    }
    if (data.length === 0) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.NoData, {
            children: formatMessage({
                id: 'content-manager.widget.last-edited.no-data',
                defaultMessage: 'No edited entries'
            })
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsx(RecentDocumentsTable, {
        documents: data,
        type: "edited",
        dynamicMaxWidth: dynamicMaxWidth
    });
};
/* -------------------------------------------------------------------------------------------------
 * LastPublishedWidget
 * -----------------------------------------------------------------------------------------------*/ const LastPublishedWidget = ({ columnWidth = 6 })=>{
    const { formatMessage } = reactIntl.useIntl();
    const { data, isLoading, error } = homepage.useGetRecentDocumentsQuery({
        action: 'publish'
    });
    const dynamicMaxWidth = calculateDynamicMaxWidth(columnWidth);
    if (isLoading) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.Loading, {});
    }
    if (error || !data) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.Error, {});
    }
    if (data.length === 0) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.NoData, {
            children: formatMessage({
                id: 'content-manager.widget.last-published.no-data',
                defaultMessage: 'No published entries'
            })
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsx(RecentDocumentsTable, {
        documents: data,
        type: "published",
        dynamicMaxWidth: dynamicMaxWidth
    });
};
/* -------------------------------------------------------------------------------------------------
 * ChartEntriesWidget
 * -----------------------------------------------------------------------------------------------*/ const RADIUS = 80;
const STROKE = 10;
const CIRCUMFERENCE = 2 * Math.PI * (RADIUS - STROKE / 2);
const ArcChart = styledComponents.styled.circle`
  stroke: ${({ theme, $arcColor })=>theme.colors[$arcColor]};
`;
const TextChart = styledComponents.styled.tspan`
  text-transform: lowercase;
  fill: ${({ theme, $textColor })=>theme.colors[$textColor]};
`;
const KeyChartItem = styledComponents.styled(designSystem.Flex)`
  width: 100%;

  ${({ theme })=>theme.breakpoints.small} {
    width: auto;
  }
`;
const DonutChartSVG = ({ data })=>{
    const { locale } = reactIntl.useIntl();
    const { formatMessage } = reactIntl.useIntl();
    const total = data.reduce((acc, curr)=>acc + curr.count, 0);
    const [tooltip, setTooltip] = React__namespace.useState({
        visible: false,
        x: 0,
        y: 0,
        value: null
    });
    let cumulativePercent = 0;
    const handleMouseOver = (e, value)=>{
        setTooltip({
            visible: true,
            x: e.clientX,
            y: e.clientY,
            value
        });
    };
    const handleMouseOut = ()=>{
        setTooltip((t)=>({
                ...t,
                visible: false
            }));
    };
    const handleFocus = (e, value)=>{
        setTooltip({
            visible: true,
            x: e.currentTarget.getBoundingClientRect().width / 2 + e.currentTarget.getBoundingClientRect().left,
            y: e.currentTarget.getBoundingClientRect().height + e.currentTarget.getBoundingClientRect().top,
            value
        });
    };
    return /*#__PURE__*/ jsxRuntime.jsxs(designSystem.Flex, {
        direction: "column",
        gap: 6,
        margin: "auto",
        children: [
            /*#__PURE__*/ jsxRuntime.jsxs("svg", {
                width: RADIUS * 2,
                height: RADIUS * 2,
                viewBox: `0 0 ${RADIUS * 2} ${RADIUS * 2}`,
                style: {
                    position: 'relative',
                    zIndex: 1
                },
                children: [
                    /*#__PURE__*/ jsxRuntime.jsx("g", {
                        transform: `rotate(-90 ${RADIUS} ${RADIUS})`,
                        children: data.map((value)=>{
                            const percent = value.count / total * 100;
                            const arcLength = percent / 100 * CIRCUMFERENCE;
                            const dashArray = `${arcLength} ${CIRCUMFERENCE - arcLength}`;
                            const dashOffset = CIRCUMFERENCE * (1 - cumulativePercent / 100);
                            const el = /*#__PURE__*/ jsxRuntime.jsx(ArcChart, {
                                cx: RADIUS,
                                cy: RADIUS,
                                r: RADIUS - STROKE / 2,
                                fill: "none",
                                strokeWidth: STROKE,
                                strokeDasharray: dashArray,
                                strokeDashoffset: dashOffset,
                                style: {
                                    transition: 'stroke-dashoffset 0.3s',
                                    cursor: 'pointer'
                                },
                                tabIndex: 0,
                                "aria-describedby": tooltip.visible ? 'chart-tooltip' : undefined,
                                onFocus: (e)=>handleFocus(e, value),
                                onBlur: handleMouseOut,
                                onMouseMove: (e)=>handleMouseOver(e, value),
                                onMouseLeave: handleMouseOut,
                                $arcColor: value.color
                            }, value.label);
                            cumulativePercent += percent;
                            return el;
                        })
                    }),
                    /*#__PURE__*/ jsxRuntime.jsxs("text", {
                        x: RADIUS,
                        y: RADIUS,
                        textAnchor: "middle",
                        fontSize: "2.4rem",
                        fontWeight: "bold",
                        children: [
                            /*#__PURE__*/ jsxRuntime.jsx(TextChart, {
                                x: RADIUS,
                                dy: "0",
                                $textColor: "neutral800",
                                children: new Intl.NumberFormat(locale, {
                                    notation: 'compact',
                                    maximumFractionDigits: 1
                                }).format(total)
                            }),
                            /*#__PURE__*/ jsxRuntime.jsx(TextChart, {
                                x: RADIUS,
                                dy: "1.4em",
                                fontSize: "1.4rem",
                                fontWeight: "normal",
                                $textColor: "neutral600",
                                children: formatMessage({
                                    id: 'content-manager.widget.chart-entries.count.label',
                                    defaultMessage: '{count, plural, =0 {entries} one {entry} other {entries}}'
                                }, {
                                    count: total
                                })
                            })
                        ]
                    })
                ]
            }),
            tooltip.visible && tooltip.value && /*#__PURE__*/ jsxRuntime.jsx(designSystem.Portal, {
                style: {
                    position: 'fixed',
                    left: 16,
                    top: 16,
                    zIndex: 2,
                    transform: `translate(${tooltip.x}px, ${tooltip.y}px)`
                },
                "data-testid": "entries-chart-tooltip",
                children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                    background: "neutral900",
                    padding: 2,
                    borderRadius: 1,
                    textAlign: "center",
                    role: "tooltip",
                    "aria-live": "polite",
                    children: /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                        textColor: "neutral0",
                        children: formatMessage({
                            id: 'content-manager.widget.chart-entries.tooltip',
                            defaultMessage: '{count} items'
                        }, {
                            count: tooltip.value.count,
                            label: tooltip.value.label
                        })
                    })
                })
            }),
            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Flex, {
                gap: 4,
                wrap: "wrap",
                children: data.map((value)=>value.count > 0 && /*#__PURE__*/ jsxRuntime.jsxs(KeyChartItem, {
                        gap: 1,
                        children: [
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Box, {
                                background: value.color,
                                padding: 2,
                                borderRadius: 1
                            }),
                            /*#__PURE__*/ jsxRuntime.jsx(designSystem.Typography, {
                                variant: "pi",
                                children: value.label
                            })
                        ]
                    }, value.label))
            })
        ]
    });
};
const ChartEntriesWidget = ()=>{
    const { formatMessage } = reactIntl.useIntl();
    const { data: countDocuments, isLoading, error } = strapiAdmin.useGetCountDocumentsQuery();
    if (isLoading) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.Loading, {});
    }
    if (error) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.Error, {});
    }
    const { draft, published, modified } = countDocuments ?? {
        draft: 0,
        published: 0,
        modified: 0
    };
    const total = draft + published + modified;
    if (!total) {
        return /*#__PURE__*/ jsxRuntime.jsx(strapiAdmin.Widget.NoData, {
            children: formatMessage({
                id: 'content-manager.widget.last-published.no-data',
                defaultMessage: 'No published entries'
            })
        });
    }
    return /*#__PURE__*/ jsxRuntime.jsx(designSystem.Flex, {
        minHeight: "100%",
        children: /*#__PURE__*/ jsxRuntime.jsx(DonutChartSVG, {
            data: [
                {
                    label: formatMessage({
                        id: 'content-manager.containers.List.draft',
                        defaultMessage: 'Draft'
                    }),
                    count: draft,
                    color: 'secondary500'
                },
                {
                    label: formatMessage({
                        id: 'content-manager.containers.List.modified',
                        defaultMessage: 'Modified'
                    }),
                    count: modified,
                    color: 'alternative500'
                },
                {
                    label: formatMessage({
                        id: 'content-manager.containers.List.published',
                        defaultMessage: 'Published'
                    }),
                    count: published,
                    color: 'success500'
                }
            ]
        })
    });
};

exports.ChartEntriesWidget = ChartEntriesWidget;
exports.LastEditedWidget = LastEditedWidget;
exports.LastPublishedWidget = LastPublishedWidget;
//# sourceMappingURL=Widgets.js.map
