import { jsx, Fragment, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { Form, Layouts, InputRenderer, useForm, BackButton } from '@strapi/admin/strapi-admin';
import { Main, Flex, Typography, Grid, Divider, Button } from '@strapi/design-system';
import { generateNKeysBetween } from 'fractional-indexing';
import pipe from 'lodash/fp/pipe';
import { useIntl } from 'react-intl';
import { ATTRIBUTE_TYPES_THAT_CANNOT_BE_MAIN_FIELD } from '../../constants/attributes.mjs';
import { capitalise } from '../../utils/strings.mjs';
import { getTranslation } from '../../utils/translations.mjs';
import { Fields, TEMP_FIELD_NAME } from './Fields.mjs';

const ConfigurationForm = ({ attributes, fieldSizes, layout: editLayout, onSubmit })=>{
    const { components, settings, layout, metadatas } = editLayout;
    const { formatMessage } = useIntl();
    const initialValues = React.useMemo(()=>{
        const transformations = pipe(flattenPanels, replaceMainFieldWithNameOnly, extractMetadata, addTmpSpaceToLayout, addTmpKeysToLayout);
        return {
            layout: transformations(layout),
            settings
        };
    }, [
        layout,
        settings
    ]);
    return /*#__PURE__*/ jsx(Fragment, {
        children: /*#__PURE__*/ jsx(Main, {
            children: /*#__PURE__*/ jsxs(Form, {
                initialValues: initialValues,
                onSubmit: onSubmit,
                method: "PUT",
                children: [
                    /*#__PURE__*/ jsx(Header, {
                        name: settings.displayName ?? ''
                    }),
                    /*#__PURE__*/ jsx(Layouts.Content, {
                        children: /*#__PURE__*/ jsxs(Flex, {
                            alignItems: "stretch",
                            background: "neutral0",
                            direction: "column",
                            gap: 6,
                            hasRadius: true,
                            shadow: "tableShadow",
                            paddingTop: 6,
                            paddingBottom: 6,
                            paddingLeft: 7,
                            paddingRight: 7,
                            children: [
                                /*#__PURE__*/ jsx(Typography, {
                                    variant: "delta",
                                    tag: "h2",
                                    children: formatMessage({
                                        id: getTranslation('containers.SettingPage.settings'),
                                        defaultMessage: 'Settings'
                                    })
                                }),
                                /*#__PURE__*/ jsxs(Grid.Root, {
                                    children: [
                                        /*#__PURE__*/ jsx(Grid.Item, {
                                            col: 6,
                                            xs: 12,
                                            direction: "column",
                                            alignItems: "stretch",
                                            children: /*#__PURE__*/ jsx(InputRenderer, {
                                                type: "enumeration",
                                                label: formatMessage({
                                                    id: getTranslation('containers.SettingPage.editSettings.entry.title'),
                                                    defaultMessage: 'Entry title'
                                                }),
                                                hint: formatMessage({
                                                    id: getTranslation('containers.SettingPage.editSettings.entry.title.description'),
                                                    defaultMessage: 'Set the display field of your entry'
                                                }),
                                                name: "settings.mainField",
                                                options: Object.entries(attributes).reduce((acc, [key, attribute])=>{
                                                    if (!attribute) {
                                                        return acc;
                                                    }
                                                    /**
                       * Create the list of attributes from the schema as to which can
                       * be our `mainField` and dictate the display name of the schema
                       * we're editing.
                       */ if (!ATTRIBUTE_TYPES_THAT_CANNOT_BE_MAIN_FIELD.includes(attribute.type)) {
                                                        acc.push({
                                                            label: key,
                                                            value: key
                                                        });
                                                    }
                                                    return acc;
                                                }, [])
                                            })
                                        }),
                                        /*#__PURE__*/ jsx(Grid.Item, {
                                            paddingTop: 6,
                                            paddingBottom: 6,
                                            col: 12,
                                            s: 12,
                                            direction: "column",
                                            alignItems: "stretch",
                                            children: /*#__PURE__*/ jsx(Divider, {})
                                        }),
                                        /*#__PURE__*/ jsx(Grid.Item, {
                                            col: 12,
                                            s: 12,
                                            direction: "column",
                                            alignItems: "stretch",
                                            children: /*#__PURE__*/ jsx(Typography, {
                                                variant: "delta",
                                                tag: "h3",
                                                children: formatMessage({
                                                    id: getTranslation('containers.SettingPage.view'),
                                                    defaultMessage: 'View'
                                                })
                                            })
                                        }),
                                        /*#__PURE__*/ jsx(Grid.Item, {
                                            col: 12,
                                            s: 12,
                                            direction: "column",
                                            alignItems: "stretch",
                                            children: /*#__PURE__*/ jsx(Fields, {
                                                attributes: attributes,
                                                components: components,
                                                fieldSizes: fieldSizes,
                                                metadatas: metadatas
                                            })
                                        })
                                    ]
                                })
                            ]
                        })
                    })
                ]
            })
        })
    });
};
/**
 * @internal
 * @description Panels don't exist in the layout, so we flatten by one.
 */ const flattenPanels = (layout)=>layout.flat(1);
/**
 * @internal
 * @description We don't need the mainField object in the layout, we only need the name.
 */ const replaceMainFieldWithNameOnly = (layout)=>layout.map((row)=>row.map((field)=>({
                ...field,
                mainField: field.mainField?.name
            })));
/**
 * @internal
 * @description We extract the metadata values from the field layout, because these are editable by the user.
 */ const extractMetadata = (layout)=>{
    return layout.map((row)=>row.map(({ label, disabled, hint, placeholder, size, name, mainField })=>({
                label,
                editable: !disabled,
                description: hint,
                mainField,
                placeholder,
                size,
                name,
                __temp_key__: ''
            })));
};
/**
 * @internal
 * @description Each row of the layout has a max size of 12 (based on bootstrap grid system)
 * So in order to offer a better drop zone we add the _TEMP_ div to complete the remaining substract (12 - existing)
 */ const addTmpSpaceToLayout = (layout)=>[
        ...layout.map((row)=>{
            const totalSpaceTaken = row.reduce((acc, field)=>acc + field.size, 0);
            if (totalSpaceTaken < 12) {
                return [
                    ...row,
                    {
                        name: TEMP_FIELD_NAME,
                        size: 12 - totalSpaceTaken,
                        __temp_key__: ''
                    }
                ];
            }
            return row;
        })
    ];
/**
 * @internal
 * @description At this point of the transformations we have Field[][], but each row for the form should have a __temp_key__
 * applied. This means we need to change it so `Field` is nested under the children property.
 */ const addTmpKeysToLayout = (layout)=>{
    const keys = generateNKeysBetween(undefined, undefined, layout.length);
    return layout.map((row, rowIndex)=>{
        const fieldKeys = generateNKeysBetween(undefined, undefined, row.length);
        return {
            __temp_key__: keys[rowIndex],
            children: row.map((field, fieldIndex)=>{
                return {
                    ...field,
                    __temp_key__: fieldKeys[fieldIndex]
                };
            })
        };
    });
};
const Header = ({ name })=>{
    const { formatMessage } = useIntl();
    const modified = useForm('Header', (state)=>state.modified);
    const isSubmitting = useForm('Header', (state)=>state.isSubmitting);
    return /*#__PURE__*/ jsx(Layouts.Header, {
        title: formatMessage({
            id: getTranslation('components.SettingsViewWrapper.pluginHeader.title'),
            defaultMessage: `Configure the view - {name}`
        }, {
            name: capitalise(name)
        }),
        subtitle: formatMessage({
            id: getTranslation('components.SettingsViewWrapper.pluginHeader.description.edit-settings'),
            defaultMessage: 'Customize how the edit view will look like.'
        }),
        navigationAction: /*#__PURE__*/ jsx(BackButton, {}),
        primaryAction: /*#__PURE__*/ jsx(Button, {
            disabled: !modified,
            loading: isSubmitting,
            type: "submit",
            children: formatMessage({
                id: 'global.save',
                defaultMessage: 'Save'
            })
        })
    });
};

export { ConfigurationForm };
//# sourceMappingURL=Form.mjs.map
