import chalk from 'chalk';
import stringify from 'fast-safe-stringify';
import ora from 'ora';
import * as cliProgress from 'cli-progress';

const stringifyArg = (arg)=>{
    return typeof arg === 'object' ? stringify(arg) : arg;
};
const createLogger = (options = {})=>{
    const { silent = false, debug = false, timestamp = true } = options;
    const state = {
        errors: 0,
        warning: 0
    };
    return {
        get warnings () {
            return state.warning;
        },
        get errors () {
            return state.errors;
        },
        async debug (...args) {
            if (silent || !debug) {
                return;
            }
            console.log(chalk.cyan(`[DEBUG]${timestamp ? `\t[${new Date().toISOString()}]` : ''}`), ...args.map(stringifyArg));
        },
        info (...args) {
            if (silent) {
                return;
            }
            console.info(chalk.blue(`[INFO]${timestamp ? `\t[${new Date().toISOString()}]` : ''}`), ...args.map(stringifyArg));
        },
        log (...args) {
            if (silent) {
                return;
            }
            console.info(chalk.blue(`${timestamp ? `\t[${new Date().toISOString()}]` : ''}`), ...args.map(stringifyArg));
        },
        success (...args) {
            if (silent) {
                return;
            }
            console.info(chalk.green(`[SUCCESS]${timestamp ? `\t[${new Date().toISOString()}]` : ''}`), ...args.map(stringifyArg));
        },
        warn (...args) {
            state.warning += 1;
            if (silent) {
                return;
            }
            console.warn(chalk.yellow(`[WARN]${timestamp ? `\t[${new Date().toISOString()}]` : ''}`), ...args.map(stringifyArg));
        },
        error (...args) {
            state.errors += 1;
            if (silent) {
                return;
            }
            console.error(chalk.red(`[ERROR]${timestamp ? `\t[${new Date().toISOString()}]` : ''}`), ...args.map(stringifyArg));
        },
        // @ts-expect-error – returning a subpart of ora is fine because the types tell us what is what.
        spinner (text) {
            if (silent) {
                return {
                    succeed () {
                        return this;
                    },
                    fail () {
                        return this;
                    },
                    start () {
                        return this;
                    },
                    text: '',
                    isSpinning: false
                };
            }
            return ora(text);
        },
        progressBar (totalSize, text) {
            if (silent) {
                return {
                    start () {
                        return this;
                    },
                    stop () {
                        return this;
                    },
                    update () {
                        return this;
                    }
                };
            }
            const progressBar = new cliProgress.SingleBar({
                format: `${text ? `${text} |` : ''}${chalk.green('{bar}')}| {percentage}%`,
                barCompleteChar: '\u2588',
                barIncompleteChar: '\u2591',
                hideCursor: true,
                forceRedraw: true
            });
            progressBar.start(totalSize, 0);
            return progressBar;
        }
    };
};

export { createLogger };
//# sourceMappingURL=logger.mjs.map
