'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var passport = require('koa-passport');
var index = require('../../utils/index.js');
var utils = require('./utils.js');
var sessionAuth = require('../../../../../shared/utils/session-auth.js');

const defaultConnectionError = ()=>new Error('Invalid connection payload');
const authenticate = async (ctx, next)=>{
    const { params: { provider } } = ctx;
    const redirectUrls = utils.default.getPrefixedRedirectUrls();
    // @ts-expect-error - can not use null to authenticate
    return passport.authenticate(provider, null, async (error, profile)=>{
        if (error || !profile || !profile.email) {
            if (error) {
                strapi.log.error(error);
            }
            strapi.eventHub.emit('admin.auth.error', {
                error: error || defaultConnectionError(),
                provider
            });
            return ctx.redirect(redirectUrls.error);
        }
        const user = await index.getService('user').findOneByEmail(profile.email);
        const scenario = user ? existingUserScenario : nonExistingUserScenario;
        return scenario(ctx, next)(user || profile, provider);
    })(ctx, next);
};
const existingUserScenario = (ctx, next)=>async (user, provider)=>{
        const redirectUrls = utils.default.getPrefixedRedirectUrls();
        if (!user.isActive) {
            strapi.eventHub.emit('admin.auth.error', {
                error: new Error(`Deactivated user tried to login (${user.id})`),
                provider
            });
            return ctx.redirect(redirectUrls.error);
        }
        ctx.state.user = user;
        return next();
    };
const nonExistingUserScenario = (ctx, next)=>async (profile, provider)=>{
        const { email, firstname, lastname, username } = profile;
        const redirectUrls = utils.default.getPrefixedRedirectUrls();
        const adminStore = await utils.default.getAdminStore();
        const { providers } = await adminStore.get({
            key: 'auth'
        });
        // We need at least the username or the firstname/lastname combination to register a new user
        const isMissingRegisterFields = !username && (!firstname || !lastname);
        if (!providers.autoRegister || !providers.defaultRole || isMissingRegisterFields) {
            strapi.eventHub.emit('admin.auth.error', {
                error: defaultConnectionError(),
                provider
            });
            return ctx.redirect(redirectUrls.error);
        }
        const defaultRole = await index.getService('role').findOne({
            id: providers.defaultRole
        });
        // If the default role has been misconfigured, redirect with an error
        if (!defaultRole) {
            strapi.eventHub.emit('admin.auth.error', {
                error: defaultConnectionError(),
                provider
            });
            return ctx.redirect(redirectUrls.error);
        }
        // Register a new user with the information given by the provider and login with it
        ctx.state.user = await index.getService('user').create({
            email,
            username,
            firstname,
            lastname,
            roles: [
                defaultRole.id
            ],
            isActive: true,
            registrationToken: null
        });
        strapi.eventHub.emit('admin.auth.autoRegistration', {
            user: ctx.state.user,
            provider
        });
        return next();
    };
const redirectWithAuth = async (ctx)=>{
    const { params: { provider } } = ctx;
    const redirectUrls = utils.default.getPrefixedRedirectUrls();
    const { user } = ctx.state;
    try {
        const sessionManager = sessionAuth.getSessionManager();
        if (!sessionManager) {
            strapi.log.error('SessionManager not available for SSO authentication');
            return ctx.redirect(redirectUrls.error);
        }
        const userId = String(user.id);
        const deviceId = sessionAuth.generateDeviceId();
        const { token: refreshToken, absoluteExpiresAt } = await sessionManager('admin').generateRefreshToken(userId, deviceId, {
            type: 'refresh'
        });
        const cookieOptions = sessionAuth.buildCookieOptionsWithExpiry('refresh', absoluteExpiresAt);
        ctx.cookies.set(sessionAuth.REFRESH_COOKIE_NAME, refreshToken, cookieOptions);
        const accessResult = await sessionManager('admin').generateAccessToken(refreshToken);
        if ('error' in accessResult) {
            strapi.log.error('Failed to generate access token for SSO user');
            return ctx.redirect(redirectUrls.error);
        }
        const { token: accessToken } = accessResult;
        const configuredSecure = strapi.config.get('admin.auth.cookie.secure');
        const isProduction = process.env.NODE_ENV === 'production';
        const isSecure = typeof configuredSecure === 'boolean' ? configuredSecure : isProduction;
        const domain = strapi.config.get('admin.auth.domain');
        ctx.cookies.set('jwtToken', accessToken, {
            httpOnly: false,
            secure: isSecure,
            overwrite: true,
            domain
        });
        const sanitizedUser = index.getService('user').sanitizeUser(user);
        strapi.eventHub.emit('admin.auth.success', {
            user: sanitizedUser,
            provider
        });
        ctx.redirect(redirectUrls.success);
    } catch (error) {
        strapi.log.error('SSO authentication failed during token generation', error);
        strapi.eventHub.emit('admin.auth.error', {
            error: error instanceof Error ? error : new Error('Unknown SSO error'),
            provider
        });
        return ctx.redirect(redirectUrls.error);
    }
};
var middlewares = {
    authenticate,
    redirectWithAuth
};

exports.authenticate = authenticate;
exports.default = middlewares;
exports.redirectWithAuth = redirectWithAuth;
//# sourceMappingURL=middlewares.js.map
