import { logout } from '../reducer.mjs';
import { isFetchError, getFetchClient } from './getFetchClient.mjs';

const isAuthPath = (url)=>/^\/admin\/(login|logout|access-token)\b/.test(url);
const simpleQuery = async (query, api)=>{
    const { signal, dispatch } = api;
    const executeQuery = async (queryToExecute)=>{
        const { get, post, del, put } = getFetchClient();
        if (typeof queryToExecute === 'string') {
            const result = await get(queryToExecute, {
                signal
            });
            return result;
        }
        const { url, method = 'GET', data, config } = queryToExecute;
        if (method === 'POST') {
            return post(url, data, {
                ...config,
                signal
            });
        }
        if (method === 'DELETE') {
            return del(url, {
                ...config,
                signal
            });
        }
        if (method === 'PUT') {
            return put(url, data, {
                ...config,
                signal
            });
        }
        return get(url, {
            ...config,
            signal
        });
    };
    try {
        const result = await executeQuery(query);
        return {
            data: result.data
        };
    } catch (err) {
        // Handle error of type FetchError
        if (isFetchError(err)) {
            // If we receive a 401 here, getFetchClient already tried to refresh and failed.
            // Log the user out since their session is no longer valid.
            if (err.status === 401) {
                const url = typeof query === 'string' ? query : query.url;
                if (!isAuthPath(url)) {
                    try {
                        const { post } = getFetchClient();
                        await post('/admin/logout');
                    } catch  {
                    // no-op
                    }
                    dispatch(logout());
                }
            }
            if (typeof err.response?.data === 'object' && err.response?.data !== null && 'error' in err.response?.data) {
                /**
         * This will most likely be ApiError
         */ return {
                    data: undefined,
                    error: err.response?.data.error
                };
            } else {
                return {
                    data: undefined,
                    error: {
                        name: 'UnknownError',
                        message: err.message,
                        details: err.response,
                        status: err.status
                    }
                };
            }
        }
        const error = err;
        return {
            data: undefined,
            error: {
                name: error.name,
                message: error.message,
                stack: error.stack
            }
        };
    }
};
const fetchBaseQuery = ()=>simpleQuery;
const isBaseQueryError = (error)=>{
    return error.name !== undefined;
};

export { fetchBaseQuery, isBaseQueryError };
//# sourceMappingURL=baseQuery.mjs.map
