'use strict';

var jsxRuntime = require('react/jsx-runtime');
var React = require('react');
var reactRouterDom = require('react-router-dom');
var Context = require('../components/Context.js');
var hooks = require('../core/store/hooks.js');
var StrapiApp = require('./StrapiApp.js');
var useQueryParams = require('../hooks/useQueryParams.js');
var reducer = require('../reducer.js');
var api = require('../services/api.js');
var auth = require('../services/auth.js');
var deviceId = require('../utils/deviceId.js');
var getFetchClient = require('../utils/getFetchClient.js');

function _interopNamespaceDefault(e) {
  var n = Object.create(null);
  if (e) {
    Object.keys(e).forEach(function (k) {
      if (k !== 'default') {
        var d = Object.getOwnPropertyDescriptor(e, k);
        Object.defineProperty(n, k, d.get ? d : {
          enumerable: true,
          get: function () { return e[k]; }
        });
      }
    });
  }
  n.default = e;
  return Object.freeze(n);
}

var React__namespace = /*#__PURE__*/_interopNamespaceDefault(React);

const [Provider, useAuth] = Context.createContext('Auth');
const STORAGE_KEYS = {
    TOKEN: 'jwtToken',
    STATUS: 'isLoggedIn'
};
const AuthProvider = ({ children, _defaultPermissions = [], _disableRenewToken = false })=>{
    const dispatch = hooks.useTypedDispatch();
    const runRbacMiddleware = StrapiApp.useStrapiApp('AuthProvider', (state)=>state.rbac.run);
    const location = reactRouterDom.useLocation();
    const [{ rawQuery }] = useQueryParams.useQueryParams();
    const locationRef = React__namespace.useRef(location);
    // Update ref without causing re-render
    React__namespace.useEffect(()=>{
        locationRef.current = location;
    }, [
        location
    ]);
    const token = hooks.useTypedSelector((state)=>state.admin_app.token ?? null);
    const { data: user, isLoading: isLoadingUser } = auth.useGetMeQuery(undefined, {
        /**
     * If there's no token, we don't try to fetch
     * the user data because it will fail.
     */ skip: !token
    });
    const { data: userPermissions = _defaultPermissions, refetch, isUninitialized, isLoading: isLoadingPermissions } = auth.useGetMyPermissionsQuery(undefined, {
        skip: !token
    });
    const navigate = reactRouterDom.useNavigate();
    const [loginMutation] = auth.useLoginMutation();
    const [logoutMutation] = auth.useLogoutMutation();
    const clearStateAndLogout = React__namespace.useCallback(()=>{
        dispatch(api.adminApi.util.resetApiState());
        dispatch(reducer.logout());
        navigate('/auth/login');
    }, [
        dispatch,
        navigate
    ]);
    React__namespace.useEffect(()=>{
        if (user) {
            if (user.preferedLanguage) {
                dispatch(reducer.setLocale(user.preferedLanguage));
            }
        }
    }, [
        dispatch,
        user
    ]);
    /**
   * Register a callback to update Redux state when the token is refreshed.
   * This ensures the app state stays in sync with the token stored in localStorage/cookies.
   */ React__namespace.useEffect(()=>{
        getFetchClient.setOnTokenUpdate((newToken)=>{
            dispatch(reducer.setToken(newToken));
        });
        return ()=>{
            getFetchClient.setOnTokenUpdate(null);
        };
    }, [
        dispatch
    ]);
    React__namespace.useEffect(()=>{
        /**
     * This will log a user out of all tabs if they log out in one tab.
     */ const handleUserStorageChange = (event)=>{
            if (event.key === STORAGE_KEYS.STATUS && event.newValue === null) {
                clearStateAndLogout();
            }
        };
        window.addEventListener('storage', handleUserStorageChange);
        return ()=>{
            window.removeEventListener('storage', handleUserStorageChange);
        };
    });
    const login = React__namespace.useCallback(async ({ rememberMe, ...body })=>{
        const res = await loginMutation({
            ...body,
            deviceId: deviceId.getOrCreateDeviceId(),
            rememberMe
        });
        /**
       * There will always be a `data` key in the response
       * because if something fails, it will throw an error.
       */ if ('data' in res) {
            const { token } = res.data;
            dispatch(reducer.login({
                token,
                persist: rememberMe
            }));
        }
        return res;
    }, [
        dispatch,
        loginMutation
    ]);
    const logout = React__namespace.useCallback(async ()=>{
        await logoutMutation({
            deviceId: deviceId.getOrCreateDeviceId()
        });
        clearStateAndLogout();
    }, [
        clearStateAndLogout,
        logoutMutation
    ]);
    const refetchPermissions = React__namespace.useCallback(async ()=>{
        if (!isUninitialized) {
            await refetch();
        }
    }, [
        isUninitialized,
        refetch
    ]);
    const [checkPermissions] = auth.useLazyCheckPermissionsQuery();
    const checkUserHasPermissions = React__namespace.useCallback(async (permissions, passedPermissions, // TODO:
    // Here we have parameterised checkUserHasPermissions in order to pass
    // query context from elsewhere in the application.
    // See packages/core/content-manager/admin/src/features/DocumentRBAC.tsx
    // This is in order to calculate permissions on accurate query params.
    // We should be able to rely on the query params in this provider
    // If we need to pass additional context to the RBAC middleware
    // we should define a better context type.
    rawQueryContext)=>{
        /**
       * If there's no permissions to check, then we allow it to
       * pass to preserve existing behaviours.
       *
       * TODO: should we review this? it feels more dangerous than useful.
       */ if (!permissions || permissions.length === 0) {
            return [
                {
                    action: '',
                    subject: ''
                }
            ];
        }
        /**
       * Given the provided permissions, return the permissions from either passedPermissions
       * or userPermissions as this is expected to be the full permission entity.
       */ const actualUserPermissions = passedPermissions ?? userPermissions;
        const matchingPermissions = actualUserPermissions.filter((permission)=>permissions.findIndex((perm)=>perm.action === permission.action && // Only check the subject if it's provided
                (perm.subject == undefined || perm.subject === permission.subject)) >= 0);
        const middlewaredPermissions = await runRbacMiddleware({
            user,
            permissions: userPermissions,
            pathname: locationRef.current.pathname,
            search: (rawQueryContext || rawQuery).split('?')[1] ?? ''
        }, matchingPermissions);
        const shouldCheckConditions = middlewaredPermissions.some((perm)=>Array.isArray(perm.conditions) && perm.conditions.length > 0);
        if (!shouldCheckConditions) {
            return middlewaredPermissions;
        }
        const { data, error } = await checkPermissions({
            permissions: middlewaredPermissions.map((perm)=>({
                    action: perm.action,
                    subject: perm.subject
                }))
        });
        if (error) {
            throw error;
        } else {
            return middlewaredPermissions.filter((_, index)=>data?.data[index] === true);
        }
    }, [
        checkPermissions,
        rawQuery,
        runRbacMiddleware,
        user,
        userPermissions
    ]);
    const isLoading = isLoadingUser || isLoadingPermissions;
    return /*#__PURE__*/ jsxRuntime.jsx(Provider, {
        token: token,
        user: user,
        login: login,
        logout: logout,
        permissions: userPermissions,
        checkUserHasPermissions: checkUserHasPermissions,
        refetchPermissions: refetchPermissions,
        isLoading: isLoading,
        children: children
    });
};

exports.AuthProvider = AuthProvider;
exports.STORAGE_KEYS = STORAGE_KEYS;
exports.useAuth = useAuth;
//# sourceMappingURL=Auth.js.map
