import { jsx, jsxs } from 'react/jsx-runtime';
import { useId, useState, useRef, useEffect } from 'react';
import { SubNav as SubNav$1, Flex, Box, Typography, Badge, IconButton, ScrollArea } from '@strapi/design-system';
import { Plus, ChevronDown } from '@strapi/icons';
import { NavLink } from 'react-router-dom';
import { styled } from 'styled-components';
import { HEIGHT_TOP_NAVIGATION, HEIGHT_TOP_NAVIGATION_MEDIUM } from '../constants/theme.mjs';
import { tours } from './GuidedTour/Tours.mjs';

const MainSubNav = styled(SubNav$1)`
  width: 100%;
  height: calc(100dvh - ${HEIGHT_TOP_NAVIGATION} - 1px);
  overflow: hidden;
  background-color: ${({ theme })=>theme.colors.neutral0};
  display: flex;
  flex-direction: column;
  border-right: 0;
  box-shadow: none;
  position: fixed;
  top: calc(${HEIGHT_TOP_NAVIGATION} + 1px);
  left: 0;
  z-index: 2;

  ${({ theme })=>theme.breakpoints.medium} {
    width: 23.2rem;
    position: sticky;
    top: 0;
    border-right: 1px solid ${({ theme })=>theme.colors.neutral150};
  }
  ${({ theme })=>theme.breakpoints.large} {
    height: 100dvh;
  }
`;
const Main = ({ children, ...props })=>/*#__PURE__*/ jsx(MainSubNav, {
        ...props,
        children: children
    });
const StyledLink = styled(NavLink)`
  display: flex;
  align-items: center;
  justify-content: space-between;
  text-decoration: none;
  height: 32px;

  color: ${({ theme })=>theme.colors.neutral800};

  &.active > div {
    ${({ theme })=>{
    return `
        background-color: ${theme.colors.primary100};
        color: ${theme.colors.primary700};
        font-weight: 500;
      `;
}}
  }

  &:hover.active > div {
    ${({ theme })=>{
    return `
        background-color: ${theme.colors.primary100};
      `;
}}
  }

  &:hover > div {
    ${({ theme })=>{
    return `
        background-color: ${theme.colors.neutral100};
      `;
}}
  }

  &:focus-visible {
    outline-offset: -2px;
  }
`;
const Link = (props)=>{
    const { label, endAction, handleClick, ...rest } = props;
    return /*#__PURE__*/ jsx(StyledLink, {
        ...rest,
        onClick: handleClick,
        children: /*#__PURE__*/ jsx(Box, {
            width: '100%',
            paddingLeft: 3,
            paddingRight: 3,
            borderRadius: 1,
            children: /*#__PURE__*/ jsxs(Flex, {
                justifyContent: "space-between",
                width: "100%",
                gap: 1,
                children: [
                    /*#__PURE__*/ jsx(Typography, {
                        tag: "div",
                        lineHeight: "32px",
                        width: {
                            initial: '80dvw',
                            medium: '100%'
                        },
                        overflow: "hidden",
                        style: {
                            textOverflow: 'ellipsis',
                            whiteSpace: 'nowrap'
                        },
                        children: label
                    }),
                    /*#__PURE__*/ jsx(Flex, {
                        gap: 2,
                        children: endAction
                    })
                ]
            })
        })
    });
};
const StyledHeader = styled(Flex)`
  flex: 0 0 ${HEIGHT_TOP_NAVIGATION};
  height: ${HEIGHT_TOP_NAVIGATION};

  ${({ theme })=>theme.breakpoints.medium} {
    flex: 0 0 ${HEIGHT_TOP_NAVIGATION_MEDIUM};
    height: ${HEIGHT_TOP_NAVIGATION_MEDIUM};
  }
`;
const Header = ({ label })=>{
    return /*#__PURE__*/ jsx(StyledHeader, {
        justifyContent: "space-between",
        paddingLeft: {
            initial: 4,
            large: 5
        },
        paddingRight: {
            initial: 4,
            large: 5
        },
        children: /*#__PURE__*/ jsx(Typography, {
            variant: "beta",
            tag: "h2",
            children: label
        })
    });
};
const Sections = ({ children, ...props })=>{
    return /*#__PURE__*/ jsx(Box, {
        paddingTop: 4,
        paddingBottom: 4,
        maxWidth: {
            initial: '100%',
            medium: '23.2rem'
        },
        children: /*#__PURE__*/ jsx(Flex, {
            tag: "ul",
            gap: "5",
            direction: "column",
            alignItems: "stretch",
            ...props,
            children: children.map((child, index)=>{
                return /*#__PURE__*/ jsx("li", {
                    children: child
                }, index);
            })
        })
    });
};
/**
 * TODO:
 * This would be better in the content-type-builder package directly but currently
 * the SubNav API does not expose a way to wrap the link, instead it wraps the link and the list
 */ const GuidedTourTooltip = ({ sectionId, children })=>{
    switch(sectionId){
        case 'models':
            return /*#__PURE__*/ jsx(tours.contentTypeBuilder.CollectionTypes, {
                children: /*#__PURE__*/ jsx(tours.contentTypeBuilder.YourTurn, {
                    children: children
                })
            });
        case 'singleTypes':
            return /*#__PURE__*/ jsx(tours.contentTypeBuilder.SingleTypes, {
                children: children
            });
        case 'components':
            return /*#__PURE__*/ jsx(tours.contentTypeBuilder.Components, {
                children: children
            });
        default:
            return children;
    }
};
const Section = ({ label, children, link, sectionId, badgeLabel })=>{
    const listId = useId();
    return /*#__PURE__*/ jsxs(Flex, {
        direction: "column",
        alignItems: "stretch",
        gap: 2,
        children: [
            /*#__PURE__*/ jsx(Box, {
                paddingLeft: {
                    initial: 4,
                    large: 5
                },
                paddingRight: {
                    initial: 4,
                    large: 5
                },
                children: /*#__PURE__*/ jsxs(Flex, {
                    position: "relative",
                    justifyContent: "space-between",
                    gap: 2,
                    children: [
                        /*#__PURE__*/ jsx(Flex, {
                            children: /*#__PURE__*/ jsx(Box, {
                                paddingRight: 1,
                                children: /*#__PURE__*/ jsx(Typography, {
                                    variant: "sigma",
                                    textColor: "neutral600",
                                    children: label
                                })
                            })
                        }),
                        /*#__PURE__*/ jsxs(Flex, {
                            gap: 1,
                            children: [
                                badgeLabel && /*#__PURE__*/ jsx(Badge, {
                                    backgroundColor: "neutral150",
                                    textColor: "neutral600",
                                    children: badgeLabel
                                }),
                                link && /*#__PURE__*/ jsx(GuidedTourTooltip, {
                                    sectionId: sectionId,
                                    children: /*#__PURE__*/ jsx(IconButton, {
                                        label: link.label,
                                        variant: "ghost",
                                        withTooltip: true,
                                        onClick: link.onClick,
                                        size: "XS",
                                        children: /*#__PURE__*/ jsx(Plus, {})
                                    })
                                })
                            ]
                        })
                    ]
                })
            }),
            /*#__PURE__*/ jsx(Flex, {
                tag: "ol",
                id: listId,
                direction: "column",
                gap: "2px",
                alignItems: 'stretch',
                marginLeft: {
                    initial: 1,
                    large: 2
                },
                marginRight: {
                    initial: 1,
                    large: 2
                },
                children: children.map((child, index)=>{
                    return /*#__PURE__*/ jsx("li", {
                        children: child
                    }, index);
                })
            })
        ]
    });
};
const SubSectionHeader = styled.button`
  cursor: pointer;
  width: 100%;
  border: none;
  padding: 0;
  background: transparent;
  display: flex;
  align-items: center;
  border-radius: ${({ theme })=>theme.borderRadius};
  padding-left: ${({ theme })=>theme.spaces[3]};
  padding-right: ${({ theme })=>theme.spaces[3]};
  padding-top: ${({ theme })=>theme.spaces[2]};
  padding-bottom: ${({ theme })=>theme.spaces[2]};

  &:hover {
    background-color: ${({ theme })=>theme.colors.neutral100};
  }
`;
const SubSectionLinkWrapper = styled.li`
  ${StyledLink} > div {
    padding-left: 36px;
  }
`;
const SubSection = ({ label, children })=>{
    const [isOpen, setOpenLinks] = useState(true);
    const [contentHeight, setContentHeight] = useState(0);
    const listId = useId();
    const contentRef = useRef(null);
    useEffect(()=>{
        if (contentRef.current) {
            setContentHeight(contentRef.current.scrollHeight);
        }
    }, [
        children
    ]);
    const handleClick = ()=>{
        setOpenLinks((prev)=>!prev);
    };
    return /*#__PURE__*/ jsxs(Box, {
        children: [
            /*#__PURE__*/ jsx(Flex, {
                justifyContent: "space-between",
                children: /*#__PURE__*/ jsxs(SubSectionHeader, {
                    onClick: handleClick,
                    "aria-expanded": isOpen,
                    "aria-controls": listId,
                    children: [
                        /*#__PURE__*/ jsx(ChevronDown, {
                            "aria-hidden": true,
                            fill: "neutral500",
                            style: {
                                transform: `rotate(${isOpen ? '0deg' : '-90deg'})`,
                                transition: 'transform 0.5s'
                            }
                        }),
                        /*#__PURE__*/ jsx(Box, {
                            paddingLeft: 2,
                            children: /*#__PURE__*/ jsx(Typography, {
                                tag: "span",
                                fontWeight: "semiBold",
                                textColor: "neutral800",
                                children: label
                            })
                        })
                    ]
                })
            }),
            /*#__PURE__*/ jsx(Flex, {
                ref: contentRef,
                tag: "ul",
                id: listId,
                direction: "column",
                gap: "2px",
                alignItems: 'stretch',
                style: {
                    maxHeight: isOpen ? `${contentHeight}px` : 0,
                    overflow: 'hidden',
                    transition: 'max-height 0.5s cubic-bezier(0, 1, 0, 1)'
                },
                children: children.map((child, index)=>{
                    return /*#__PURE__*/ jsx(SubSectionLinkWrapper, {
                        children: child
                    }, index);
                })
            })
        ]
    });
};
const PageWrapper = styled(Box)`
  ${MainSubNav} {
    background-color: transparent;
    border-right: none;
  }

  ${({ theme })=>theme.breakpoints.medium} {
    ${MainSubNav} {
      top: 0;
    }
  }
`;
const Content = ({ children })=>{
    return /*#__PURE__*/ jsx(ScrollArea, {
        children: children
    });
};
const SubNav = {
    Main,
    Content,
    Header,
    Link,
    Sections,
    Section,
    SubSection,
    PageWrapper
};

export { SubNav };
//# sourceMappingURL=SubNav.mjs.map
