import { jsx, jsxs } from 'react/jsx-runtime';
import * as React from 'react';
import { Box, Flex, Typography, useCallbackRef } from '@strapi/design-system';
import { HEIGHT_TOP_NAVIGATION, RESPONSIVE_DEFAULT_SPACING } from '../../constants/theme.mjs';
import { useDeviceType } from '../../hooks/useDeviceType.mjs';
import { useElementOnScreen } from '../../hooks/useElementOnScreen.mjs';

const BaseHeaderLayout = /*#__PURE__*/ React.forwardRef(({ navigationAction, primaryAction, secondaryAction, subtitle, title, sticky, width, ...props }, ref)=>{
    const isSubtitleString = typeof subtitle === 'string';
    if (sticky) {
        return /*#__PURE__*/ jsx(Box, {
            display: "flex",
            paddingLeft: 6,
            paddingRight: 6,
            paddingTop: 2,
            paddingBottom: 2,
            position: "fixed",
            top: 0,
            background: "neutral0",
            shadow: "tableShadow",
            width: `${width}px`,
            zIndex: 2,
            minHeight: HEIGHT_TOP_NAVIGATION,
            "data-strapi-header-sticky": true,
            children: /*#__PURE__*/ jsxs(Flex, {
                alignItems: "center",
                justifyContent: "space-between",
                wrap: "wrap",
                width: "100%",
                children: [
                    /*#__PURE__*/ jsxs(Flex, {
                        children: [
                            navigationAction && /*#__PURE__*/ jsx(Box, {
                                paddingRight: 3,
                                children: navigationAction
                            }),
                            /*#__PURE__*/ jsxs(Box, {
                                children: [
                                    /*#__PURE__*/ jsx(Typography, {
                                        variant: "beta",
                                        tag: "h1",
                                        ...props,
                                        children: title
                                    }),
                                    isSubtitleString ? /*#__PURE__*/ jsx(Typography, {
                                        variant: "pi",
                                        textColor: "neutral600",
                                        children: subtitle
                                    }) : subtitle
                                ]
                            }),
                            secondaryAction ? /*#__PURE__*/ jsx(Box, {
                                paddingLeft: 4,
                                children: secondaryAction
                            }) : null
                        ]
                    }),
                    /*#__PURE__*/ jsx(Flex, {
                        children: primaryAction ? /*#__PURE__*/ jsx(Box, {
                            paddingLeft: 2,
                            children: primaryAction
                        }) : undefined
                    })
                ]
            })
        });
    }
    return /*#__PURE__*/ jsxs(Box, {
        ref: ref,
        paddingLeft: RESPONSIVE_DEFAULT_SPACING,
        paddingRight: RESPONSIVE_DEFAULT_SPACING,
        paddingBottom: {
            initial: 4,
            large: 8
        },
        paddingTop: {
            initial: 4,
            large: navigationAction ? 6 : 8
        },
        background: "neutral100",
        "data-strapi-header": true,
        children: [
            /*#__PURE__*/ jsxs(Flex, {
                direction: "column",
                alignItems: "initial",
                gap: 2,
                children: [
                    navigationAction,
                    /*#__PURE__*/ jsxs(Flex, {
                        justifyContent: "space-between",
                        wrap: "wrap",
                        gap: 4,
                        children: [
                            /*#__PURE__*/ jsxs(Flex, {
                                minWidth: 0,
                                children: [
                                    /*#__PURE__*/ jsx(Typography, {
                                        tag: "h1",
                                        variant: "alpha",
                                        ...props,
                                        style: {
                                            wordBreak: 'break-word',
                                            overflowWrap: 'break-word',
                                            maxWidth: '100%'
                                        },
                                        children: title
                                    }),
                                    secondaryAction ? /*#__PURE__*/ jsx(Box, {
                                        paddingLeft: 4,
                                        children: secondaryAction
                                    }) : null
                                ]
                            }),
                            primaryAction
                        ]
                    })
                ]
            }),
            isSubtitleString ? /*#__PURE__*/ jsx(Typography, {
                variant: "epsilon",
                textColor: "neutral600",
                tag: "p",
                paddingTop: {
                    initial: 4,
                    large: 0
                },
                children: subtitle
            }) : subtitle
        ]
    });
});
const HeaderLayout = (props)=>{
    const baseHeaderLayoutRef = React.useRef(null);
    const [headerSize, setHeaderSize] = React.useState(null);
    const [isVisible, setIsVisible] = React.useState(true);
    const deviceType = useDeviceType();
    const containerRef = useElementOnScreen(setIsVisible, {
        root: null,
        rootMargin: '0px',
        threshold: 0
    });
    useResizeObserver([
        containerRef,
        baseHeaderLayoutRef
    ], ()=>{
        const headerContainer = baseHeaderLayoutRef.current ?? containerRef.current;
        if (headerContainer) {
            const newSize = headerContainer.getBoundingClientRect();
            setHeaderSize((prevSize)=>{
                // Only update if size actually changed
                if (!prevSize || prevSize.height !== newSize.height || prevSize.width !== newSize.width) {
                    return newSize;
                }
                return prevSize;
            });
        }
    });
    React.useEffect(()=>{
        if (baseHeaderLayoutRef.current || containerRef.current) {
            const headerContainer = baseHeaderLayoutRef.current ?? containerRef.current;
            setHeaderSize(headerContainer?.getBoundingClientRect() ?? null);
        }
    }, [
        containerRef
    ]);
    if (deviceType === 'mobile') {
        return /*#__PURE__*/ jsx(BaseHeaderLayout, {
            ...props
        });
    }
    return /*#__PURE__*/ jsxs("div", {
        ref: containerRef,
        children: [
            /*#__PURE__*/ jsx("div", {
                style: {
                    height: headerSize?.height
                },
                children: isVisible && /*#__PURE__*/ jsx(BaseHeaderLayout, {
                    ref: baseHeaderLayoutRef,
                    ...props
                })
            }),
            !isVisible && /*#__PURE__*/ jsx(BaseHeaderLayout, {
                ...props,
                sticky: true,
                width: headerSize?.width
            })
        ]
    });
};
HeaderLayout.displayName = 'HeaderLayout';
/**
 * useResizeObserver: hook that observes the size of an element and calls a callback when it changes.
 */ const useResizeObserver = (sources, onResize)=>{
    const handleResize = useCallbackRef(onResize);
    React.useLayoutEffect(()=>{
        const resizeObs = new ResizeObserver(handleResize);
        if (Array.isArray(sources)) {
            sources.forEach((source)=>{
                if (source.current) {
                    resizeObs.observe(source.current);
                }
            });
        } else if (sources.current) {
            resizeObs.observe(sources.current);
        }
        return ()=>{
            resizeObs.disconnect();
        };
    }, [
        sources,
        handleResize
    ]);
};

export { BaseHeaderLayout, HeaderLayout };
//# sourceMappingURL=HeaderLayout.mjs.map
